import assign from 'lodash.assign';

const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, PanelRow, ToggleControl } = wp.components;
const { addFilter } = wp.hooks;
const { __ } = wp.i18n;
const { dispatch } = wp.data;

/**
 * Add layout control attribute to block.
 *
 * @param {object} settings Current block settings.
 * @param {string} name Name of block.
 *
 * @returns {object} Modified block settings.
 */
const addAttributes = (settings, name) => {
  // Use Lodash's assign to gracefully handle if attributes are undefined
  settings.attributes = assign(settings.attributes, {
    responsiveBlockControl: {
      type: 'object',
      default: {
        mobile: true,
        tablet: true,
        desktop: true,
        wide: true,
      },
    },
  });

  return settings;
};

addFilter('blocks.registerBlockType', 'responsiveBlockControl/attribute', addAttributes);

//fine prima parte
/**
 * Create HOC to add layout control to inspector controls of block.
 */
const withVisibilityControl = createHigherOrderComponent((BlockEdit) => {
  return (props) => {
    const {
      clientId,
      attributes,
      isSelected,
    } = props;

    const {
      responsiveBlockControl,
    } = attributes;

    const handleChange = (device) => {
      const newValue = !responsiveBlockControl[device];

      delete responsiveBlockControl[device];

      const blockOptions = Object.assign({ [device]: newValue }, responsiveBlockControl);

      dispatch('core/block-editor').updateBlockAttributes(clientId, { responsiveBlockControl: blockOptions });
    };

    if (typeof responsiveBlockControl === 'undefined') {
      attributes.responsiveBlockControl = [];
    }

    return (
      <Fragment>
        <BlockEdit {...props} />
        {isSelected &&
          <InspectorControls>
            <PanelBody
              title={__('Responsive Manager', 'thor')}
              initialOpen={false}
            >
              <PanelRow>
                <ToggleControl
                  label={__('Hide on Mobile', 'thor')}
                  checked={typeof responsiveBlockControl.mobile !== 'undefined' && !responsiveBlockControl.mobile}
                  onChange={() => handleChange('mobile')}
                />
              </PanelRow>
              <PanelRow>
                <ToggleControl
                  label={__('Hide on Tablet', 'thor')}
                  checked={typeof responsiveBlockControl.tablet !== 'undefined' && !responsiveBlockControl.tablet}
                  onChange={() => handleChange('tablet')}
                />
              </PanelRow>
              <PanelRow>
                <ToggleControl
                  label={__('Hide on Desktop', 'thor')}
                  checked={typeof responsiveBlockControl.desktop !== 'undefined' && !responsiveBlockControl.desktop}
                  onChange={() => handleChange('desktop')}
                />
              </PanelRow>
              <PanelRow>
                <ToggleControl
                  label={__('Hide on Wide', 'thor')}
                  checked={typeof responsiveBlockControl.wide !== 'undefined' && !responsiveBlockControl.wide}
                  onChange={() => handleChange('wide')}
                />
              </PanelRow>
            </PanelBody>
          </InspectorControls>
        }
      </Fragment>
    );
  };
}, 'withVisibilityControl');


addFilter('editor.BlockEdit', 'responsiveBlockControl/control', withVisibilityControl);

const updateVisibilityClasses = (classString, responsiveBlockControl) => {
  const visibilityClasses = [];
  let classArray = [];

  if (typeof responsiveBlockControl.mobile !== 'undefined' && !responsiveBlockControl.mobile) {
    visibilityClasses.push('thor-hidden-sm');
  }
  if (typeof responsiveBlockControl.tablet !== 'undefined' && !responsiveBlockControl.tablet) {
    visibilityClasses.push('thor-hidden-md');
  }
  if (typeof responsiveBlockControl.desktop !== 'undefined' && !responsiveBlockControl.desktop) {
    visibilityClasses.push('thor-hidden-lg');
  }
  if (typeof responsiveBlockControl.wide !== 'undefined' && !responsiveBlockControl.wide) {
    visibilityClasses.push('thor-hidden-xl');
  }

  // remove all layout classes
  if (typeof classString !== 'undefined') {
    classArray = classString.trim().split(' '); // remove whitespace and create array
    classArray = classArray.filter((value) => {
      return !value.startsWith('thor-hidden-');
    });
  }

  // add current visibility classes
  const newArray = classArray.concat(visibilityClasses);

  return newArray.join(' ');
};

/**
 * Add custom element class in save element.
 * This adds the classes to the saved element.
 *
 * @param {Object} extraProps     Block element.
 * @param {Object} blockType      Blocks object.
 * @param {Object} attributes     Blocks attributes.
 *
 * @return {Object} extraProps Modified block element.
 */
function addExtraClass(extraProps, blockType, attributes) {
  const { responsiveBlockControl } = attributes;

  if (typeof responsiveBlockControl === 'undefined') {
    return extraProps;
  }

  const newProps = extraProps;

  newProps.className = updateVisibilityClasses(extraProps.className, responsiveBlockControl);

  return newProps;
}

addFilter(
  'blocks.getSaveContent.extraProps',
  'responsiveBlockControl/applyExtraClass',
  addExtraClass
);